<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

header('Content-Type: application/json');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Function to log debug information
function debug_log($message, $data = null) {
    $log = date('Y-m-d H:i:s') . " - " . $message;
    if ($data !== null) {
        $log .= " - Data: " . print_r($data, true);
    }
    error_log($log);
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    debug_log("User not logged in");
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit();
}

// Get parameters from request
$product_id = isset($_GET['product_id']) ? intval($_GET['product_id']) : 0;
$serial_number = isset($_GET['serial_number']) ? trim($_GET['serial_number']) : '';

debug_log("Check specific serial request", [
    'product_id' => $product_id,
    'serial_number' => $serial_number
]);

if ($product_id <= 0 || empty($serial_number)) {
    debug_log("Invalid parameters");
    echo json_encode([
        'success' => false, 
        'message' => 'Invalid product ID or serial number'
    ]);
    exit();
}

try {
    // Check if the serial number exists and is available for this product
    $query = "
        SELECT psn.id, psn.serial_number, psn.status, p.name as product_name
        FROM product_serial_numbers psn
        JOIN products p ON psn.product_id = p.id
        WHERE psn.product_id = ? AND psn.serial_number = ?
        LIMIT 1
    ";
    
    debug_log("Executing serial check query", [
        'query' => $query,
        'product_id' => $product_id,
        'serial_number' => $serial_number
    ]);

    $stmt = $pdo->prepare($query);
    $stmt->execute([$product_id, $serial_number]);
    $serial = $stmt->fetch(PDO::FETCH_ASSOC);

    debug_log("Serial check result", $serial);

    if ($serial) {
        if ($serial['status'] === 'available') {
            echo json_encode([
                'success' => true,
                'available' => true,
                'message' => 'Serial number is available',
                'serial_number' => $serial['serial_number'],
                'product_name' => $serial['product_name']
            ]);
        } else {
            echo json_encode([
                'success' => true,
                'available' => false,
                'message' => 'Serial number is not available (status: ' . $serial['status'] . ')',
                'serial_number' => $serial['serial_number'],
                'product_name' => $serial['product_name']
            ]);
        }
    } else {
        // Check if serial exists for any product
        $stmt2 = $pdo->prepare("
            SELECT psn.serial_number, p.name as product_name
            FROM product_serial_numbers psn
            JOIN products p ON psn.product_id = p.id
            WHERE psn.serial_number = ?
            LIMIT 1
        ");
        $stmt2->execute([$serial_number]);
        $otherProduct = $stmt2->fetch(PDO::FETCH_ASSOC);
        
        if ($otherProduct) {
            echo json_encode([
                'success' => true,
                'available' => false,
                'message' => 'Serial number belongs to a different product: ' . $otherProduct['product_name'],
                'serial_number' => $serial_number
            ]);
        } else {
            echo json_encode([
                'success' => true,
                'available' => false,
                'message' => 'Serial number not found in system',
                'serial_number' => $serial_number
            ]);
        }
    }
} catch (PDOException $e) {
    debug_log("Database error", $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} 